
// PIC16F18146
/* TODO:
 * UI modes, see below
 * Decode GPS signals
 * support CC LEDs
 */

/*
 * At start-up, A/B change GPS baud rate *
 * A short press: cycle colour scheme
 * B short press: change second 'hand'
 * A long press: time zone offset mode *
 * B long press: brightness adjustment mode *
 * In time zone offset mode: *
 * * A short press: -15 minutes *
 * * B short press: +15 minutes *
 * * A long press: return to clock *
 * * B long press: toggle 1 hour offset *
 * In brightness adjustment mode: *
 * * A short press: dimmer *
 * * B short press: brighter *
 * * A long press: return to clock *
 * * B long press: adjust LDR sensitivity *
 */

#include "config.h"
#include <xc.h>
#include <string.h>

typedef struct { unsigned char A, B, C; } port;
typedef struct { unsigned char hour, min, sec, subsec; } time;
typedef enum { black = 0, red = 1, blue = 2, mauve = 3, green = 4, yellow = 5, cyan = 6, white = 7 } colour;
typedef enum { matching_subsec = 0, white_subsec = 1, no_subsec = 2, no_sec = 3 } second_hand_config;
typedef enum { CA, CC } RGB_LED_type;
typedef struct { colour hour, min, sec, subsec; } scheme;

unsigned char cur_state = 0, update_state = 0, which_used = 0;
port anode_states[2][8];
port cathode_states[2][8];
unsigned char brightness = 128, b1down = 0, b2down = 0, b1wasdown, b2wasdown, have_am_pm = 0;
unsigned short button_press_counter, maxb1press, maxb2press, b1shortpress, b1longpress, b2shortpress, b2longpress;
__eeprom unsigned char which_colour_scheme;
__eeprom second_hand_config second_hand_cfg = matching_subsec;

#define PIN0 (1<<0)
#define PIN1 (1<<1)
#define PIN2 (1<<2)
#define PIN3 (1<<3)
#define PIN4 (1<<4)
#define PIN5 (1<<5)
#define PIN6 (1<<6)
#define PIN7 (1<<7)

#define LED_AM 60
#define LED_PM 61

void get_led_mask(unsigned char LEDnum, colour LEDc, port* pAnode, port* pCathode) {
  pAnode->A = 0;
  pAnode->B = 0;
  pAnode->C = 0;
  pCathode->A = 0;
  pCathode->B = 0;
  pCathode->C = 0;
  if( LEDnum == LED_AM ) {
    pAnode->C = PIN6;
    pCathode->C = PIN4;
  } else if( LEDnum == LED_PM ) {
    pAnode->C = PIN6;
    pCathode->C = PIN5;
  } else {
    unsigned char a = LEDnum>>2;
    unsigned char c;

    if( a < 8 )
      pAnode->C = (unsigned char)(1<<a);
    else if( a < 12 )
      pAnode->B = (unsigned char)(1<<(a-4));
    else
      pAnode->A = (unsigned char)(1<<(a-12));

    for( c = 0; c < 3; ++c ) {
      if( LEDc & 1 ) {
        unsigned char b = (unsigned char)((LEDnum&3) + ((LEDnum&3)<<1) + c + a + 1);

        while( b > 14 )
          b -= 15;

        if( b < 8 )
          pCathode->C |= (unsigned char)(1<<b);
        else if( b < 12 )
          pCathode->B |= (unsigned char)(1<<(b-4));
        else
          pCathode->A |= (unsigned char)(1<<(b-12));
      }
      LEDc = LEDc >> 1;
    }
  }
}

void advance_time(time* t) {
    if( ++t->subsec == 61 ) {
        t->subsec = 0;
        if( ++t->sec == 60 ) {
            t->sec = 0;
            if( ++t->min == 60 ) {
                t->min = 0;
                if( ++t->hour == 13 )
                    t->hour = 1;
            }
        }
    }
}

void get_led_masks_from_time(port* pAnodes, port* pCathodes, time* pTim, scheme* pSch, second_hand_config cfg) {
  unsigned char hour_hand_pos = pTim->hour, subsec_hand_pos = pTim->sec + pTim->subsec;

  while( hour_hand_pos > 11 )
    hour_hand_pos -= 12;
  while( subsec_hand_pos > 60 )
    subsec_hand_pos -= 60;
  hour_hand_pos = (unsigned char)((hour_hand_pos<<2) + hour_hand_pos + pTim->min / 12);

  get_led_mask(hour_hand_pos, pSch->hour, &pAnodes[0], &pCathodes[0]);
  get_led_mask(pTim->min, pSch->min, &pAnodes[1], &pCathodes[1]);
  switch(cfg) {
      case matching_subsec:
          get_led_mask(pTim->sec, pSch->sec, &pAnodes[2], &pCathodes[2]);
          get_led_mask(subsec_hand_pos, pSch->sec, &pAnodes[3], &pCathodes[3]);
          break;
      case white_subsec:
          get_led_mask(pTim->sec, pSch->sec, &pAnodes[2], &pCathodes[2]);
          get_led_mask(subsec_hand_pos, white, &pAnodes[3], &pCathodes[3]);
          break;
      case no_subsec:
          get_led_mask(pTim->sec, pSch->sec, &pAnodes[2], &pCathodes[2]);
      case no_sec:
          pAnodes[3].A = 0;
          pAnodes[3].B = 0;
          pAnodes[3].C = 0;
          pCathodes[3].A = 0;
          pCathodes[3].B = 0;
          pCathodes[3].C = 0;
          break;
  }
  if( cfg == no_sec ) {
    pAnodes[2].A = 0;
    pAnodes[2].B = 0;
    pAnodes[2].C = 0;
    pCathodes[2].A = 0;
    pCathodes[2].B = 0;
    pCathodes[2].C = 0;
    pAnodes[3].A = 0;
  }
  get_led_mask(pTim->hour >= 12 ? LED_PM : LED_AM, white, &pAnodes[4], &pCathodes[4]);
//  get_led_mask(pTim->hour > 12 ? LED_PM : LED_AM, red, &A, &K);
}

void initADC(void){
    ADCON0=0;           //reset ADC, single ended
    ADCON1=0;
    ADCON2=0;           //legacy mode
    ADCON3=0;
    ADPRE=0;
    ADACQ=0;
    ADCAP=0;            //extra capacitance
    ADCON0bits.CS=1;    //ADCRC
    ADCON0bits.ADFM0=1; //right justify
    ADREF = 0;          //VDD
    ADCON0bits.ADON=1;  //turn on, ready
}

unsigned short getADC(unsigned char c){
    ADPCH=c;
    ADCON0bits.GO=1;
    while(ADCON0bits.GO){}    
    return ADRES;
}

unsigned char ADC_channel_for_port(port* p) {
    unsigned char ret = 58, val = 0;
    if( p->A ) {
        ret = 0;
        val = p->A;
    } else if( p->B ) {
        ret = 8;
        val = p->B;
    } else if( p->C ) {
        ret = 16;
        val = p->C;
    }
    while( val > 1 ) {
        ++ret;
        val >>= 1;
    }
    return ret;
}

unsigned char have_LED(unsigned char LEDnum, colour LEDc, RGB_LED_type type) {
    unsigned char ret = 0;
    unsigned short ADCval;
    port A, C;
    get_led_mask(LEDnum, LEDc, &A, &C);
    if( type == CA ) {
        WPUA = A.A;
        WPUB = A.B;
        WPUC = A.C;
        LATA = 0;
        LATB = 0;
        LATC = 0;
        TRISA = ~C.A;
        TRISB = ~C.B;
        TRISC = ~C.C;
        ANSELA |= A.A;
        ANSELB |= A.B;
        ANSELC |= A.C;
        ADCval = getADC(ADC_channel_for_port(&A));
        ANSELA &= ~A.A;
        ANSELB &= ~A.B;
        ANSELC &= ~A.C;
        ret = (ADCval < 3900 && ADCval > 1024);
    } else {
        WPUA = C.A;
        WPUB = C.B;
        WPUC = C.C;
        LATA = 0xFF;
        LATB = 0xFF;
        LATC = 0xFF;
        TRISA = ~A.A;
        TRISB = ~A.B;
        TRISC = ~A.C;
        ANSELA |= C.A;
        ANSELB |= C.B;
        ANSELC |= C.C;
        ADCval = getADC(ADC_channel_for_port(&C));
        ANSELA &= ~C.A;
        ANSELB &= ~C.B;
        ANSELC &= ~C.C;
        ret = (ADCval < 3900 && ADCval > 1024);
    }
    TRISA = 0xFF;
    TRISB = 0xFF;
    TRISC = 0xFF;
    WPUA = 0;
    WPUB = 0;
    WPUC = 0;
    return ret;
}

void get_colour_scheme(scheme* ps, unsigned char which) {
    switch(which>>1) {
        case 0:
            ps->sec = red;
            break;
        case 1:
            ps->sec = green;
            break;
        case 2:
        default:
            ps->sec = blue;
            break;
    }
    switch(ps->sec) {
        case red:
            if( which&1 ) {
                ps->min = blue;
                ps->hour = green;
            } else {
                ps->min = green;
                ps->hour = blue;
            }
            break;
        case green:
            if( which&1 ) {
                ps->min = blue;
                ps->hour = red;
            } else {
                ps->min = red;
                ps->hour = blue;
            }
            break;
        case blue:
            if( which&1 ) {
                ps->min = green;
                ps->hour = red;
            } else {
                ps->min = red;
                ps->hour = green;
            }
            break;
        default:
            break;
    }
    ps->subsec = white;
}

void main(void) {
    time now = { 12, 5, 0 };
    scheme sch;
    unsigned char i, bootup = 62/*0*/, bootup_delay = 0;
    unsigned short ldr, last_ldr = 0;

    get_colour_scheme(&sch, which_colour_scheme);
    
    ANSELA = PIN5;
    ANSELB = 0;
    ANSELC = 0;
    initADC();

    have_am_pm = have_LED(LED_AM, white, CA) || have_LED(LED_PM, white, CA);

    T0CON1 = 0b01000010; // FOSC/4, prescaler = 4
    T0CON0 = 0b10000000; // on, 8-bit timer, no postscaler
    TMR0H = 0xFF;
    PIR0bits.TMR0IF = 0;
    while( !PIR0bits.TMR0IF )
        ;
    PIR0bits.TMR0IF = 0;
    PIE0bits.TMR0IE = 1;
    INTCONbits.GIE = 1;
  
    T1CON = 0b00000011;
    T1CLK = 1;
    PIR1bits.TMR1IF = 0;
    T2CLKCON = 1; // FOSC/4
    T2CON = 0b10100000; // on, prescaler = 4
    PIR2bits.TMR2IF = 0;
    PIE2bits.TMR2IE = 1;
    INTCONbits.PEIE = 1;

    while(1) {
        if( bootup < 62 ) {
            get_led_mask(bootup, white, &anode_states[which_used^1][0], &cathode_states[which_used^1][0]);
            get_led_mask(bootup, white, &anode_states[which_used^1][1], &cathode_states[which_used^1][1]);
            get_led_mask(bootup, white, &anode_states[which_used^1][2], &cathode_states[which_used^1][2]);
            get_led_mask(bootup, white, &anode_states[which_used^1][3], &cathode_states[which_used^1][3]);
            get_led_mask(bootup, white, &anode_states[which_used^1][4], &cathode_states[which_used^1][4]);
            if( ++bootup_delay == 20 ) {
                bootup_delay = 0;
                ++bootup;
            }
            T2PR = 255;
        } else {
            get_led_masks_from_time(anode_states[which_used^1], cathode_states[which_used^1], &now, &sch, second_hand_cfg);
        }
        for( i = 0; i < 5; ++i ) {
            cathode_states[which_used^1][i].A = ~(anode_states[which_used^1][i].A|cathode_states[which_used^1][i].A);
            cathode_states[which_used^1][i].B = ~(anode_states[which_used^1][i].B|cathode_states[which_used^1][i].B);
            cathode_states[which_used^1][i].C = ~(anode_states[which_used^1][i].C|cathode_states[which_used^1][i].C);
        }
        which_used ^= 1;//update_state = 1;

        while( !PIR1bits.TMR1IF )
            ;
        PIR1bits.TMR1IF = 0;

        if( bootup < 62 )
            continue;

        advance_time(&now);
        ldr = getADC(5);// LDR is on RA5
        b1down = ldr > 4096-32;
        b2down = ldr < 384;
        if( !b1down && !b2down )
            last_ldr = (last_ldr + ldr)>>1;
        ldr = last_ldr;
        if( ldr > 2048+512 )
            ldr -= 2048+512;
        else
            ldr = 0;
        ldr >>= 2;
        if( ldr > 255 )
            ldr = 255;
        brightness = (unsigned char)ldr;
        if( brightness < 16 )
            brightness = 16;
        T2PR = brightness;

        if( b1down ) {
            if( !b1wasdown ) {
                b1wasdown = 1;
                button_press_counter = 0;
            } else {
                if( ++button_press_counter > maxb1press )
                    maxb1press = button_press_counter;
            }
        } else if( b2down ) {
            if( !b2wasdown ) {
                b2wasdown = 1;
                button_press_counter = 0;
            } else {
                if( ++button_press_counter > maxb2press )
                    maxb2press = button_press_counter;
            }
        } else {
            if( b1wasdown ) {
                b1wasdown = 0;
                button_press_counter = 0;
            }
            if( b2wasdown ) {
                b2wasdown = 0;
                button_press_counter = 0;
            }
            if( button_press_counter == 8 ) {
                if( maxb1press > 8 ) {
                    if( maxb1press > 30 ) {
                        ++b1longpress;
                    } else {
                        ++b1shortpress;
                    }
                }
                maxb1press = 0;
                if( maxb2press > 8 ) {
                    if( maxb2press > 30 ) {
                        ++b2longpress;
                    } else {
                        ++b2shortpress;
                    }
                }
                maxb2press = 0;
            } else {
                ++button_press_counter;
            }
        }

        if( b1shortpress ) {
            --b1shortpress;
            which_colour_scheme = (which_colour_scheme+1) & 7;
            get_colour_scheme(&sch, which_colour_scheme);
/*
            if( ++now.min == 60 ) {
                now.min = 0;
                if( ++now.hour == 13 )
                    now.hour = 1;
            }
*/
        }
        if( b2shortpress ) {
            --b2shortpress;
            if( second_hand_cfg == no_sec )
                second_hand_cfg = matching_subsec;
            else
                ++second_hand_cfg;
        }
/*
            if( now.min == 0 ) {
                now.min = 59;
                if( now.hour == 1 )
                    now.hour = 13;
                else
                    --now.hour;
            } else {
                --now.min;
            }
        }
*/
    }
}

void __interrupt() isr(){
    if( PIR0bits.TMR0IF/* && PIE0bits.TMR0IE*/ ) {
        TMR2 = 0;
        PIR2bits.TMR2IF = 0;
        if( ++cur_state == (4 + have_am_pm) )
            cur_state = 0;
        TRISA = 0xFF;
        TRISB = 0xFF;
        TRISC = 0xFF;
        LATA = anode_states[which_used][cur_state].A;
        LATB = anode_states[which_used][cur_state].B;
        LATC = anode_states[which_used][cur_state].C;
        TRISA = cathode_states[which_used][cur_state].A;
        TRISB = cathode_states[which_used][cur_state].B;
        TRISC = cathode_states[which_used][cur_state].C;
        PIR0bits.TMR0IF = 0;
    }
    if( PIR2bits.TMR2IF/* && PIE2bits.TMR2IE*/ ) {
        TRISA = 0xFF;
        TRISB = 0xFF;
        TRISC = 0xFF;
        PIR2bits.TMR2IF = 0;
    }
}
